<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\AppCode;
use App\Models\Bidding;
use App\Models\Contract;
use App\Models\Media;
use App\Models\TenancyContract;
use App\Models\User;
use DB;
use Session;
use Auth;
use Cache;
use Excel;
use DateTime;
use DateInterval;
use PDF;
use Mail;
class ContractsController extends MailsController
{
    /**
     * Display a listing of the successfully commission paid bids.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $bid_status_id = array('51','52','53','54','64');
        $landlord_id = Auth::user()->id;
        $bidding_list = Bidding::with(['property:id,name,address,landlord_id,title_deed'],
                                        'tenant:id,firstname,lastname,email,address,country_code,mobile','employmentstatus:id,name','bidstatus:id,name','contract:id,path')
                                ->whereIn('bid_status_id',$bid_status_id)
                                ->whereHas('Property', function($q) use ($landlord_id){
                                     $q->where('landlord_id', $landlord_id);
                                 })
                                ->paginate(10);
      
        return view('contracts/index', compact('bidding_list'));
    }

    /**
     * Display a listing of the successfully commission paid tenant bids.
     *
     * @return \Illuminate\Http\Response
     */
    public function tenantContract(Request $request)
    {
        $bid_status_id = array('51','52','53','54','64');
        $tenant_id = Auth::user()->id;
        $bidding_list = Bidding::with(array('Tenant:id,firstname,lastname,email,address,country_code,mobile',
                                    'employmentstatus:id,name',
                                    'bidstatus:id,name',
                                    'Property'=>function($query){
                                        $query->with('Landlord:id,firstname,lastname,email,address,country_code,mobile');
                                    },
                                ))/*
                                with(['property:id,name,address,landlord_id,title_deed'],
                                        'tenant:id,firstname,lastname,email,address,country_code,mobile','employmentstatus:id,name','bidstatus:id,name','contract:id,path')*/
                                ->whereIn('bid_status_id',$bid_status_id)
                                ->where('tenant_id',$tenant_id)
                                ->paginate(10);

        return view('contracts/tenant-contract', compact('bidding_list'));
    }

    /**
     * updating bid status
     *
     * @return \Illuminate\Http\Response
     */
    public function bidStatusUpdate(Request $request)
    {
        $bid_id = $request->input('bid_id');
        if(!empty($bid_id)){
            $bid_status_id = $request->input('bid_status_id');
            $user_id = $request->input('tenant_id');
            /* Sending Notification to the bidder */
            $user_detail = User::findOrFail($user_id);

            $post['bid_id'] = $bid_id;
            $post['property_id'] = '';
            $post['type_id'] = env('MESSAGE_TYPE_BID');

            $post['message'] = 'Owner has updated your bid status against property '.$request->input('property_name');
            $user_detail->notify(new \App\Notifications\PostNewNotification($post));
           
            try{
                $bidding = new Bidding;
                $bidding->where('id',$bid_id)
                        ->update(['bid_status_id'=>$bid_status_id]); //Updating bid status to contract inprogress/pending documents
                Session::flash('Success','Bid Status updated successfully');    
            }
            catch(Exception $e){
                Session::flash('Error','Something went wrong !');       
            }
        }
        else{
            Session::flash('Error','Something went wrong !');       
        }
        
        return redirect()->back();
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try{
            $data = $this->getData($request); //Validating request data
            
            $bid_id = $data['bid_id'];
            $document = $data['document'];
            $imageName = 'Image_'.rand().'.'.$document->getClientOriginalExtension();
            $path = public_path('media/contract/'.$bid_id);
            $document->move($path, $imageName);
           
            $data['path'] = 'media/contract/'.$bid_id.'/'.$imageName;

            $contract = new Contract;
            $contract->fill($data);
            $contract->save();
            $contract_id = $contract->id;
            
            $bid = new Bidding;
            $bid->where('id',$bid_id)
                ->update(['bid_status_id'=>54, 'contract_id'=>$contract_id]);

            Session::flash('Success','Contract added successfully');

            return redirect()->back();
        }
        catch(Exception $exception){
            return back()->withInput()
                         ->withErrors(['unexpected_error' => 'Unexpected error occurred while trying to process your request!']);
        }
        
    }
    /**
     * Get the request's data from the request.
     *
     * @param Illuminate\Http\Request\Request $request 
     * @return array
     */
    protected function getData(Request $request)
    {
        $rules = [
            'document' => 'required|mimes:pdf,doc,docx|max:2048',
            'bid_id' =>'required|nullable',
            'tenant_id' =>'required|nullable',
            'property_id' =>'required|nullable',
            'landlord_id' =>'required|nullable',
        ];
        $data = $request->validate($rules);
        return $data;
    }
	
	/**
     * Function that let landlord fill contract form
     *
     * @param 
     *
     * @return \Illuminate\Http\Response
     */
    public function addContract( $bid_id = null)
    {
		if(isset($bid_id)){
			$bid_id = base64_decode($bid_id);			
			
			//$bidding_detail = Bidding::findOrFail($bid_id);
			$bidding_detail = Bidding::where('id', $bid_id)
					->with(array(
						'Property'=>function($query){
                            $query->with('PropertyTerm:id,term_id,property_id', 'User:id,firstname,lastname,email,country_code,mobile','PropertyType:id,name');
                        },
                        'Tenant:id,firstname,lastname,email,country_code,mobile'
					))
                    ->first()
                    ->toArray();

			
			$terms_conditions = AppCode::termsCondition()->orderBy('sort_order')->pluck('name','id')->all();
		} else{
            Session::flash('Error','Unauthorised access !');            
            return redirect('/');
        }
		//compact('contracts','tenant_id','landlord_id','start_date','end_date','tenants','landlords')
		return view('contracts/contract', compact('bidding_detail','terms_conditions'));
	}

    /**
     * Function that save contract form details to database
     *
     * @param 
     *
     * @return \Illuminate\Http\Response
     */
    public function saveContract( Request $request)
    {
        $data = $request->all();
        //$data = $this->getContractData($request); //Validating request data
        
        $tenancy_contract = new TenancyContract;
        $tenancy_contract->fill($data);
        $tenancy_contract->save();
        $contract_id = $tenancy_contract->id;
        
        $bid_id = $data['bid_id'];        
        $bid = new Bidding;
            $bid->where('id',$bid_id)
                ->update(['bid_status_id'=>53, 'contract_id'=>$contract_id,'tenancy_contract_starting_date'=>date("Y-m-d", strtotime($data['contract_period_from'])), 'tenancy_contract_end_date' => date("Y-m-d", strtotime($data['contract_period_to']))]);

        $tenant_detail = User::findOrFail($data['tenant_id']);
        $tenant_name = $tenant_detail['firstname']." ".$tenant_detail['lastname'];
        $tenant_email = $tenant_detail['email'];
        
        $post['bid_id'] = $bid_id;
        $post['property_id'] = '';
        $post['contract_id'] = $contract_id;
        $post['type_id'] = env('MESSAGE_TYPE_CONTRACT');

        $post['message'] = 'Tenancy Contract has been uploaded by landlord for property'.$data['property_name'].'. Please approve the same';
        $tenant_detail->notify(new \App\Notifications\PostNewNotification($post));

        /* Notifying tenant by email about contract upload */
        $mail_result = $this->contract_upload($tenant_name, $tenant_email, $data['property_name']);

        Session::flash('Success','Contract added successfully');

        return redirect('/contracts')->withInput();        
    }
	
    /**
     * Get the request's data from the request.
     *
     * @param Illuminate\Http\Request\Request $request 
     * @return array
     */
    protected function getContractData(Request $request)
    {
        $rules = [
            'contract_date' => 'required|integer|nullable',
            'contract_month' => 'required|integer|nullable',
            'contract_year' => 'required|integer|nullable',
            'contract_number' => 'required|string|nullable',
            'property_usage' => 'required|string|nullable',
            'owner_name' => 'required|string|nullable',
            'landlord_name' => 'required|string|nullable',
            'tenant_name' => 'required|string|nullable',
            'landlord_email' => 'required|email|nullable',
            'tenant_email' => 'required|email|nullable',
            'tenant_phone' => 'required|integer|nullable',
            'landlord_phone' => 'required|integer|nullable',
            'building_name' => 'required|string|nullable',
            'building_location' => 'required|string|nullable',
            'property_size' => 'required|string|nullable',
            'property_type' => 'required|string|nullable',
            'property_number' => 'required|string|nullable',
            'premises_number' => 'required|string|nullable',
            'plot_number' => 'required|string|nullable',
            'contract_period_from' => 'required|string|nullable',
            'contract_period_to' => 'required|string|nullable',
            'annual_rent' => 'required|string|nullable',
            'contract_value' => 'required|string|nullable',
            'security_deposit' => 'required|string|nullable',
            'payment_mode' => 'required|string|nullable',
            'tenant_signature_date' => 'required|string|nullable',
        ];
        $data = $request->validate($rules);
        return $data;
    }
    /**
     * Function that view tenancy contract form
     *
     * @param 
     *
     * @return \Illuminate\Http\Response
     */
    public function viewContract( $bid_id = null)
    {
        if(isset($bid_id)){
            $bid_id = base64_decode($bid_id);

            $bidding_detail = Bidding::where('id', $bid_id)
                    ->with(array(
                        'TenancyContract',
                        'Property'=>function($query){
                            $query->with('PropertyTerm:id,term_id,property_id');
                        },
                    ))
                    ->first()
                    ->toArray();
                    
            $terms_conditions = AppCode::termsCondition()->orderBy('sort_order')->pluck('name','id')->all();
        } else{
            Session::flash('Error','Unauthorised access !');            
            return redirect('/');
        }
        return view('contracts/view-contract', compact('bidding_detail','terms_conditions'));
    }
    
    /**
     * Function that approve the contract by tenant
     *
     * @param 
     *
     * @return \Illuminate\Http\Response
     */
    public function tenantApproveContract( $tenancy_contract_id = null)
    {
        if(isset($tenancy_contract_id)){
            $contract_id = base64_decode($tenancy_contract_id);

            $contract_detail = TenancyContract::Select('id','tenant_id','landlord_id','bid_id')->where('id', $contract_id)
                                ->first()
                                ->toArray();

            $bidding_detail = Bidding::Select('id','property_id')->where('id', $contract_detail['bid_id'])
                    ->with('Property:id,name')
                    ->first()
                    ->toArray();
            $property_name = $bidding_detail['property']['name'];

            $tenancy_contract = new TenancyContract;
            $tenancy_contract->where('id',$contract_id)
                ->update(['tenant_approve_status'=>1]);

            $landlord_detail = User::findOrFail($contract_detail['landlord_id']);
            $landlord_name = $landlord_detail['firstname']." ".$landlord_detail['lastname'];
            $landlord_email = $landlord_detail['email'];
            
            $post['bid_id'] = $contract_detail['bid_id'];
            $post['property_id'] = '';
            $post['contract_id'] = $contract_id;
            $post['type_id'] = env('MESSAGE_TYPE_CONTRACT');
            $post['message'] = 'Tenant has approved the contract against property '.$property_name.'. Please accept the same';
            $landlord_detail->notify(new \App\Notifications\PostNewNotification($post));

            /* Notifying tenant by email about contract upload */
            $mail_result = $this->tenant_approve_contract($landlord_name, $landlord_email, $property_name);

            Session::flash('Success','Contract has been approved successfully.');    
            return redirect('contracts/tenant-contract');

        } else{
            Session::flash('Error','Unauthorised access !');            
            return redirect('/');
        }
    }

    /**
     * Function that accept the contract by landlord
     *
     * @param 
     *
     * @return \Illuminate\Http\Response
     */
    public function landlordAcceptContract( $tenancy_contract_id = null)
    {
        if(isset($tenancy_contract_id)){
            $contract_id = base64_decode($tenancy_contract_id);

            /* Fetching contract details with landlord details */
            $contract_detail = TenancyContract::Select('id','tenant_id','landlord_id','bid_id')->with('landlord:id,firstname,lastname,email')->where('id', $contract_id)
                                ->first()
                                ->toArray();
            /* Selecting bid_id, property_id & property_name to be set in mail notification */
            $bidding_detail = Bidding::Select('id','property_id')->where('id', $contract_detail['bid_id'])
                    ->with('Property:id,name')
                    ->first()
                    ->toArray();            
            $property_name = $bidding_detail['property']['name'];

            /* Calculating end date of contract */
            $end_date = $this->endCycle(date("Y-m-d"), 12);            

            /* Updating contract starting date & end date in bddings table */
            $bid = new Bidding;
            $bid->where('id',$contract_detail['bid_id'])
                ->update(['bid_status_id'=>64]);

            /* Updating the status of landlord acceptance */
            $tenancy_contract = new TenancyContract;
            $tenancy_contract->where('id',$contract_id)
                ->update(['landlord_accept_status'=>1]);

            /* Notifying tenant about the acceptance */
            $tenant_detail = User::findOrFail($contract_detail['tenant_id']);
            $tenant_name = $tenant_detail['firstname']." ".$tenant_detail['lastname'];
            $tenant_email = $tenant_detail['email'];            
            $post['bid_id'] = $contract_detail['bid_id'];
            $post['property_id'] = '';
            $post['contract_id'] = $contract_id;
            $post['type_id'] = env('MESSAGE_TYPE_CONTRACT');
            $post['message'] = 'Landlord has accepted the tenancy contract for property'.$property_name;
            $tenant_detail->notify(new \App\Notifications\PostNewNotification($post));

            /* Loading contract details to be set on pdf view file */            
            $bidding_detail = Bidding::where('id', $contract_detail['bid_id'])
                    ->with(array(
                        'TenancyContract',
                        'Property'=>function($query){
                            $query->with('PropertyTerm:id,term_id,property_id');
                        },
                    ))
                    ->first()
                    ->toArray();                    
            $terms_conditions = AppCode::termsCondition()->orderBy('sort_order')->pluck('name','id')->all();
            /* setting pdf name */
            $pdf_name = $contract_id.'_'.time().'.pdf';
            /* path to save the pdf */
            $dir = public_path('/media/contract/'.$contract_detail['landlord_id']);
            //$dir = $_SERVER['DOCUMENT_ROOT']."/bid2rent/public/media/contract/".$contract_detail['landlord_id'];
            if(is_dir($dir)){
                /* generating pdf from view */
                $pdf = PDF::loadView('contracts.view-contract-form',compact('data','bidding_detail','terms_conditions'));
                $pdf->save($dir.'/'.$pdf_name);
            }
            else{
                mkdir($dir);
                /* generating pdf from view */
                $pdf = PDF::loadView('contracts.view-contract-form',compact('data','bidding_detail','terms_conditions'));
                $pdf->save($dir.'/'.$pdf_name);
            }
            /* full path of pdf file */
            $pdf_path = $dir.'/'.$pdf_name; 

            $tenancy_contract->where('id',$contract_id)
                ->update(['pdf_path'=>$pdf_path]);
                
            /* sending pdf as attachment in mail to landlord*/
            $data = array('name'=>$contract_detail['landlord']['firstname'].' '.$contract_detail['landlord']['lastname'], 'email'=>$contract_detail['landlord']['email'], 'property_name'=>$property_name,'pdf_path'=>$pdf_path);
            Mail::send('contract-pdf', $data, function($message) use ($data){
                $message->to($data['email'], 'Bid2Rent')->subject
                    ('Tenancy Contract');
                $message->attach($data['pdf_path']);
                $message->from(env('MAIL_USERNAME'),'Bid2Rent');
            });

            /* sending pdf as attachment in mail to tenant*/
            $data = array('name'=>$tenant_name, 'email'=>$tenant_email, 'property_name'=>$property_name,'pdf_path'=>$pdf_path);
            Mail::send('contract-pdf', $data, function($message) use ($data){
                $message->to($data['email'], 'Bid2Rent')->subject
                    ('Tenancy Contract');
                $message->attach($data['pdf_path']);
                $message->from(env('MAIL_USERNAME'),'Bid2Rent');
            });            

            Session::flash('Success','Contract has been accepted successfully and copy has been mailed to your email id.'); 

            return redirect('/contracts');

        } else{
            Session::flash('Error','Unauthorised access !');            
            return redirect('/');
        }
    }

    /**
     * Function that download contract
     *
     * @param 
     *
     * @return \Illuminate\Http\Response
     */
    public function downloadContract( $contract_id = null)
    {
        if(isset($contract_id)){
            $contract_id = base64_decode($contract_id);

            $contract_detail = TenancyContract::Select('id','pdf_path')
                            ->where('id', $contract_id)
                            ->first()
                            ->toArray();  
            
            $pdf_path = $contract_detail['pdf_path'];

            if (file_exists($pdf_path))
            {
                header('Content-Description: File Transfer');
                header('Content-Type: application/pdf');
                header('Content-Disposition: attachment; filename='.basename($pdf_path));
                header('Expires: 0');
                header('Cache-Control: must-revalidate');
                header('Pragma: public');
                header('Content-Length: ' . filesize($pdf_path));
                ob_clean();
                flush();
                readfile($pdf_path);
                die();
            }
            else{
                Session::flash('Error','No contract file found');  
                return redirect()->back();
            }

        } else{
            Session::flash('Error','Unauthorised access !');            
            return redirect('/');
        }
    }



    /**
     * Function that edit tenancy form by landlord
     *
     * @param 
     *
     * @return \Illuminate\Http\Response
     */
    public function editContract( $contract_id = null)
    {
        if(isset($contract_id)){
            $contract_id = base64_decode($contract_id);

            $contract_detail = TenancyContract::where('id', $contract_id)
                                        ->first()
                                        ->toArray();

            $bidding_detail = Bidding::where('id', $contract_detail['bid_id'])
                    ->with(array(
                        'Property'=>function($query){
                            $query->with('PropertyTerm:id,term_id,property_id');
                        },
                    ))
                    ->first()
                    ->toArray();

            $terms_conditions = AppCode::termsCondition()->orderBy('sort_order')->pluck('name','id')->all();
        } else{
            Session::flash('Error','Unauthorised access !');            
            return redirect('/');
        }
        //compact('contracts','tenant_id','landlord_id','start_date','end_date','tenants','landlords')
        return view('contracts/edit', compact('bidding_detail','contract_detail','terms_conditions'));
    }


    /**
     * Function that update contract form details to database
     *
     * @param 
     *
     * @return \Illuminate\Http\Response
     */
    public function updateContract( Request $request)
    {
        $data = $request->all();
        //$data = $this->getContractData($request); //Validating request data

        $tenancy_contract = TenancyContract::where('id',$data['contract_id'])->first();
        $tenancy_contract->update($data);  
        
        $tenant_detail = User::findOrFail($data['tenant_id']);
        $tenant_name = $tenant_detail['firstname']." ".$tenant_detail['lastname'];
        $tenant_email = $tenant_detail['email'];
        
        $post['bid_id'] = $data['bid_id'];
        $post['property_id'] = '';
        $post['contract_id'] = $data['contract_id'];
        $post['type_id'] = env('MESSAGE_TYPE_CONTRACT');
        
        $post['message'] = 'Tenancy Contract has been updated by landlord for property'.$data['property_name'].'. Please approve the same';
        $tenant_detail->notify(new \App\Notifications\PostNewNotification($post));

        /* Notifying tenant by email about contract upload */
        $mail_result = $this->contract_upload($tenant_name, $tenant_email, $data['property_name']);

        Session::flash('Success','Contract updated successfully');

        return redirect('contracts/view_contract/'.base64_encode($data['bid_id']));        
    }


    /**
     * support function to calculate end date
     *
     * @return void
     */

      public function endCycle($end_date, $months)
        {
            $date = new DateTime($end_date);

            // call second function to add the months
            $newDate = $date->add($this->add_months($months, $date));

            // goes back 1 day from date, remove if you want same day of month
            $newDate->sub(new DateInterval('P1D'));

            //formats final date to Y-m-d form
            $dateReturned = $newDate->format('Y-m-d');

            return $dateReturned;
        }

    /**
     * support function to add specific months
     *
     * @return void
     */
      private function add_months($months, DateTime $dateObject)
        {
            $next = new DateTime($dateObject->format('Y-m-d'));
            $next->modify('last day of +'.$months.' month');

            if($dateObject->format('d') > $next->format('d')) {
                return $dateObject->diff($next);
            } else {
                return new DateInterval('P'.$months.'M');
            }
        }


    #Code for Admin Section

    /**
     * Display a list of contracts.
     *
     * @return \Illuminate\Http\Response
     */
    public function adminContracts()
    {
        $tenant_id = 0;
        $landlord_id = 0;
        $start_date = '';
        $end_date = '';

        if($_SERVER['REQUEST_METHOD']=='POST'){
            $tenant_id = $_POST['tenant_id'];
            $landlord_id = $_POST['landlord_id'];
            $start_date = (!empty($_POST['start_date'])) ? date("Y-m-d", strtotime($_POST['start_date'])) : '';
            $end_date = (!empty($_POST['end_date'])) ? date("Y-m-d", strtotime($_POST['end_date'])) : '';
        }
        /* Getting filter parameters for pagination */
        if(isset($_GET['tenant_id']) && !empty($_GET['tenant_id']))
            $tenant_id = $_GET['tenant_id'];
        if(isset($_GET['landlord_id']) && !empty($_GET['landlord_id']))
            $landlord_id = $_GET['landlord_id'];
        if(isset($_GET['start_date']) && !empty($_GET['start_date']))
            $start_date = date("Y-m-d", strtotime($_GET['start_date']));
        if(isset($_GET['end_date']) && !empty($_GET['end_date']))
            $end_date = date("Y-m-d", strtotime($_GET['end_date']));
        /* Ends Here */

        //$tenants = User::select('id','firstname','lastname','email')->where('user_type_id',env('USER_TYPE_TENANT'))->orderBy('firstname')->get()->toArray();
        $tenants = Cache::remember('tenant', env('CACHE_TIME'), function()
        {
            return User::select('id','firstname','lastname','email')->where('user_type_id',env('USER_TYPE_TENANT'))->orderBy('firstname')->get()->toArray();
        });

        //$landlords = User::select('id','firstname','lastname','email')->where('user_type_id',env('USER_TYPE_LANDLORD'))->orderBy('firstname')->get()->toArray();
        $landlords = Cache::remember('user', env('CACHE_TIME'), function()
        {
            return User::select('id','firstname','lastname','email')->where('user_type_id',env('USER_TYPE_LANDLORD'))->orderBy('firstname')->get()->toArray();
        });

        $contracts = Contract::with(array('Property','Bid','Landlord','Tenant'))
                                ->when($tenant_id, function ($query) use ($tenant_id) {
                                    return $query->where('tenant_id',$tenant_id);
                                })
                                ->when($landlord_id, function ($query) use ($landlord_id) {
                                    return $query->where('landlord_id',$landlord_id);
                                })
                                ->when($start_date, function ($query) use ($start_date,$end_date) {
                                    return $query->whereBetween('created_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                                })
                                ->paginate(10);
                                
        return view('admin/contract-list', compact('contracts','tenant_id','landlord_id','start_date','end_date','tenants','landlords'));
    }

    
    /**
     * Exporting Contracts to Excel
     *
     * @param 
     *
     * @return Illuminate\View\View
     */
    public function adminExportContracts($tenant_id, $landlord_id, $start_date=null, $end_date=null)
    {
        $conditions = array();

        if(!empty($start_date))
            $start_date = date("Y-m-d", strtotime($start_date));
        if(!empty($end_date))
            $end_date = date("Y-m-d", strtotime($end_date));

        $contracts = Contract::with(array('Property','Bid','Landlord','Tenant'))
                                ->when($tenant_id, function ($query) use ($tenant_id) {
                                    return $query->where('tenant_id',$tenant_id);
                                })
                                ->when($landlord_id, function ($query) use ($landlord_id) {
                                    return $query->where('landlord_id',$landlord_id);
                                })
                                ->when($start_date, function ($query) use ($start_date,$end_date) {
                                    return $query->whereBetween('created_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                                })
                                ->get()
                                ->toArray();
            
        Excel::create('ContractsReport', function($excel) use ($contracts){
            $excel->sheet('messages', function($sheet) use ($contracts){                
                $sheet->row(1,array('Contract ID', 'Property Name','Property address', 'Tenant Name','Landlord Name','Proposed Rent','Contract Date'));
                $i=2;
                foreach($contracts as $contract){
                    $sheet->row($i, array($contract['id'], $contract['property']['name'], $contract['property']['address'],$contract['tenant']['firstname'].' '.$contract['tenant']['lastname'], $contract['landlord']['firstname'].' '.$contract['landlord']['lastname'], $contract['bid']['propose_rent'], date("j F, Y", strtotime($contract['created_at']))));
                    $i++;
                }
            });
        })->export('xls');
        die();
    }

}
