<?php

namespace App\Http\Controllers;
use Cache;
use Excel;
use App\Models\Tenant;
use App\Models\Bidding;
use App\Models\Property;
use App\Models\BidStatus;
use App\Models\BidPriority;
use App\Models\Transaction;
use App\Models\User;
use App\Models\AppCode;
use Illuminate\Http\Request;
use App\Models\EmploymentStatus;
use App\Http\Controllers\Controller;
use App\Models\LandlordNoteVisibilityStatus;
use DB;
use Session;
use Auth;
use App\Models\Survey;
use App\Models\SmsReport;
use App\Models\Note;
class BiddingsController extends MailsController
{
    /**
     * Get the request's data from the request.
     *
     * @param Illuminate\Http\Request\Request $request 
     * @return array
     */
    protected function getData(Request $request)
    {
        $rules = [
            'property_id' => 'required|nullable',
            'tenant_id' => 'required|nullable',
            'landlord_id' => 'required|nullable',
            'monthly_rent' => 'required|nullable',
            'mobile' => 'required|nullable',
            'tenant_name' => 'required|nullable',
            'property_name' => 'required|string|min:1|nullable',
            'propose_rent' => 'required|integer|nullable',
            'move_in_date' => 'required|date_format:Y-m-d|nullable',
            'tenant_note' => 'nullable|string|min:1|nullable',
            'employment_status_id' => 'required|nullable',
            'terms' => 'required|nullable', 
            'yearly_price' => 'required|nullable'           
        ];
        $data = $request->validate($rules);
        return $data;
    }
    
    /**
     * Store a new bidding in the storage.
     *
     * @param Illuminate\Http\Request $request
     *
     * @return Illuminate\Http\RedirectResponse | Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        try {
            $data = $this->getData($request);
            
            $user_id = (!empty(Auth::user())) ? Auth::user()->id : '';
            //Array of properties against which the logged in Tenant has already made a bid
            $tenant_bid_property = Bidding::select('id','tenant_id','property_id')->where('tenant_id',$user_id)->get()->toArray();
            if(!in_array($data['property_id'], array_column($tenant_bid_property, 'property_id'))) {
                
                $data['landlord_note'] = '';
                $data['landlord_note_visibility_status_id'] = 0;
                $data['bid_status_id'] = 46;
                $data['bid_priority_id'] = 45;            
                
                $property_name = $data['property_name'];
                $proposed_rent = $data['propose_rent'];
                $monthly_rent = $data['monthly_rent'];
                $yearly_price = $data['yearly_price'];

                $yearly_price_ten_percent = ($yearly_price*10)/100;
                if($proposed_rent > ($yearly_price_ten_percent+$yearly_price)){                    
                   /* Sending Message to Owner for selecting the bid */
                    $msg_body = $data['tenant_name'].' made a bid and proposed rent is 10 percent more than the set price';
                    $recipients = trim($data['mobile']);
                    $response = Controller::SendSMS(env('SMS_API_URL'), env('SMS_USERNAME'), env('SMS_PASSWORD'), $msg_body, env('SMS_MSG_ENCODING'), env('SMS_COUNTRY_CODE'), env('SMS_AREA_CODE'), $recipients, env('SMS_HEADER')); 

                    $response_array = explode('=', $response);
                
                    $sms_data['user_id'] = $data['landlord_id'];
                    $sms_data['recipient'] = $recipients;
                    $sms_data['message'] = $msg_body;
                    $sms_data['message_id'] = '';
                    $error = substr($response_array[1],0,1);
                    # if there is not error then save SMS to sms_reports table
                    if($error == 0){                    
                        $sms_data['message_id'] = preg_replace('/(<BR>)+$/', '', $response_array[2]);
                        #Creating object for sms_report model
                        $sms_report = new SmsReport;
                        $sms_report->fill($sms_data);
                        $sms_report->save($sms_data);

                    }
                }

                $bidding = new Bidding;
                $bidding->fill($data);
                $bidding->save();



                /* Sending Notification to the Owner */
                $user_id = $data['landlord_id'];
                $user_detail = User::findOrFail($user_id);
                $landlord_email = $user_detail->email;
                $landlord_name = $user_detail->firstname.' '.$user_detail->lastname;

                $tenant_detail = User::findOrFail($data['tenant_id']);
                $tenant_name = $tenant_detail['firstname']." ".$tenant_detail['lastname'];
                
                $post['bid_id'] = '';
                $post['property_id'] = $data['property_id'];
                $post['type_id'] = env('MESSAGE_TYPE_PROPERTY');

                $post['message'] = $tenant_name.' made a bid against your property '.$data['property_name'];
                $user_detail->notify(new \App\Notifications\PostNewNotification($post));

                /* Notifying landlord by email about bid submitted */
                $mail_result = $this->bid_submit($landlord_name, $landlord_email, $tenant_name, $property_name);

                return redirect()->back()
                                 ->with('Success', 'Bidding done successfully');
            }
            else{
                return redirect()->back()
                                 ->with('Error', 'You have already bid against this property !');
            }


        } catch (Exception $exception) {

            return back()->withInput()
                         ->withErrors(['unexpected_error' => 'Unexpected error occurred while trying to process your request!']);
        }
    }
    
    
    /**
     * Display all biddings against a property
     *
     * @param int $id
     *
     * @return Illuminate\View\View
     */
    public function propertyBiddings($property_id, $landlord_id)
    {
        $property_id = base64_decode($property_id);
        $landlord_id = base64_decode($landlord_id);
        if($landlord_id == Auth::user()->id){
            $conditions = array();
            $conditions = array('property_id'=>$property_id);
            $property = Property::with(
                    array(
                        'PropertyStatus'=>function($query){
                            $query->select('id','name');
                        },
                        'Media'=>function($query){
                            $query->select('id','path','parent_id');
                        },
                        'Emirate'=>function($query){
                            $query->select('id','name');
                        },
                        'PropertyType'=>function($query){
                            $query->select('id','name');
                        },
                        'VieweingOption'=>function($query){
                            $query->select('id','name');
                        },
                        'BedroomSize'=>function($query){
                            $query->select('id','name');
                        },
                        'WashroomSize'=>function($query){
                            $query->select('id','name');
                        },
                        'User'=>function($query){
                            $query->select('id','firstname','lastname');
                        },
                        'AmenityProperty'=>function($query){
                            $query->select('id','amenity_id','property_id');
                        },
                        'Bidding'=>function($query){
                            $query->select('id','tenant_id');
                        },
                    )
                )
                ->findOrFail($property_id)
                ->toArray();

            $biddings = Bidding::with(
                    array(
                        'Tenant'=>function($query){
                            $query->select('id','firstname','lastname');
                        },
                        'EmploymentStatus'=>function($query){
                            $query->select('id','name');
                        },
                        'BidStatus'=>function($query){
                            $query->select('id','name');
                        },
                        'Note'=>function($query){
                            $query->select('id','content','visibility_id');
                        }, 
                    )
                )
                ->where($conditions)
                ->get()
                ->toArray(); 
            return view('biddings.bidding-list', compact('biddings','property'));
        }
        else{
            Session::flash('Error','Unauthorised access !');            
            return redirect('/');
        }
    }

    /**
     * Confirm a new bidding
     *
     * @param Illuminate\Http\Request $request
     *
     * @return Illuminate\Http\RedirectResponse | Illuminate\Routing\Redirector
     */
    public function confirmBidding(Request $request)
    {
        $bid_id = $request->bid_id;
        if(!empty($bid_id)){
            $bidding_detail = Bidding::with('transaction','property')
                                ->select('id','property_id','propose_rent','tenant_id','bid_status_id')
                                ->findOrFail($bid_id)
                                ->toArray();
            
            $property_id = $bidding_detail['property_id'];
            $property_name = $bidding_detail['property']['name'];
            $updated_at = date("Y-m-d H:i:s");
            
            $bidding = new Bidding;
            $bidding->where('id',$bid_id)
                    ->update(['bid_status_id'=>48,'updated_at'=>$updated_at,'bid_selection_time'=>date("Y-m-d H:i:s")]); //Setting selected bid status to Bid Confirmation Pending
            
            $bidding->where('property_id',$property_id)
                    ->where('id','!=',$bid_id)
                    ->where('bid_status_id','!=',55)
                    ->update(['bid_status_id'=>47,'updated_at'=>$updated_at]); //Setting other bid status to "On Hold"                

            $property = new Property;
            $property->where('id',$property_id)
                    ->update(['bid_selected'=>1,'updated_at'=>$updated_at]); //Updating bid_selected status against property, so that no new bid can be posted.

            /* Sending Notification to the bidder */
            $user_id = $bidding_detail['tenant_id'];
            $user_detail = User::findOrFail($user_id);

            $post['bid_id'] = $bid_id;
            $post['property_id'] = '';
            $post['type_id'] = env('MESSAGE_TYPE_BID');

            $post['message'] = 'Your bid has been selected by the Owner, Please confirm the same';
            $user_detail->notify(new \App\Notifications\PostNewNotification($post));

            $mail_result = $this->bid_select($user_detail->firstname, $user_detail->lastname, $user_detail->email, $property_name);

            Session::flash('Success','Bid selected, confirmation is pending from Tenant');
        }
        else{
            Session::flash('Success','Something went wrong !');   
        }
        return redirect()->back();
    }


    public function maxBid($property_id){
        $maxBidPrice = Bidding::selectRaw('MAX(propose_rent) as maximum_bid_price')->where('property_id',$property_id)->first()->toArray();

        echo json_encode($maxBidPrice); exit;
    }

    

    public function tenantBids(){
        $user_id = Auth::user()->id;
        $bidding_detail = Bidding::with('Property','BidStatus')
                                ->where('tenant_id',$user_id)->get()->toArray();
                               
        return view('biddings.tenant-bids', compact('bidding_detail'));
    }

    /**
     * Landlord's note on bids and its visibility
     */
    public function landlordNote(Request $request)
    {
        $request->validate(['note'=>'required','visible'=>'required','bid_id'=>'required']);
        $bid_id = $request->bid_id;
        $visibile = $request->visible;
        $note = $request->note;
        $updated_at = date("Y-m-d H:i:s");
        $bidding = Bidding::where('id',$bid_id)->first();
        $bidding->update(['landlord_note'=>$note,
                        'landlord_note_visibility_status_id'=>$visibile,
                        'updated_at'=>$updated_at]);
        
        $note_data['bid_id'] = $bid_id;
        $note_data['visibility_id'] = $visibile;
        $note_data['content'] = $note;
        $note_data['landlord_id'] = Auth::user()->id;

        $note = new Note;
        $note->fill($note_data);
        $note->save();

        Session::flash('Success','Note was successfully updated');
        return redirect()->back();
    }

    /**
     * Landlord's note on bids and its visibility
     */
    public function bidEdit(Request $request)
    {
        $request->validate(['tnote'=>'nullable',
                            'rent'=>'required',
                            'move_date'=>'required',
                            'emp_stat'=>'required',
                            'bid_id'=>'required'
                            ]);
        $bid_id = $request->bid_id;
        $note = $request->tnote;
        $rent = $request->rent;
        $move_date = $request->move_date;
        $emp_stat = $request->emp_stat;
        $updated_at = date("Y-m-d H:i:s");
        $bidding = Bidding::where('id',$bid_id)->first();
        $bidding->update(['propose_rent'=>$rent,'tenant_note'=>$note,
                        'move_in_date'=>$move_date,'employment_status_id'=>$emp_stat,
                        'updated_at'=>$updated_at]);
        
        $property_name = $request->property_name;
        $proposed_rent = $request->rent;
        $yearly_price = $request->yearly_price;
        $yearly_price_ten_percent = ($yearly_price*10)/100;

        if($proposed_rent > ($yearly_price_ten_percent+$yearly_price)){
            $tenant_name = Auth::user()->firstname.' '.Auth::user()->lastname;                   
            /* Sending Message to Owner for selecting the bid */
            $msg_body = $tenant_name.' made a bid and proposed rent is 10 percent more than the set price';
            $recipients = trim(Auth::user()->mobile);
            $response = Controller::SendSMS(env('SMS_API_URL'), env('SMS_USERNAME'), env('SMS_PASSWORD'), $msg_body, env('SMS_MSG_ENCODING'), env('SMS_COUNTRY_CODE'), env('SMS_AREA_CODE'), $recipients, env('SMS_HEADER')); 

            $response_array = explode('=', $response);
                
            $sms_data['user_id'] = $request->landlord_id;
            $sms_data['recipient'] = $recipients;
            $sms_data['message'] = $msg_body;
            $sms_data['message_id'] = '';
            $error = substr($response_array[1],0,1);
            # if there is not error then save SMS to sms_reports table
            if($error == 0){                    
                $sms_data['message_id'] = preg_replace('/(<BR>)+$/', '', $response_array[2]);
                #Creating object for sms_report model
                $sms_report = new SmsReport;
                $sms_report->fill($sms_data);
                $sms_report->save($sms_data);
            }
        }

        /* Sending Notification to the Owner */
        $landlord_id = $request->landlord_id;
        $user_detail = User::findOrFail($landlord_id);
        $landlord_email = $user_detail->email;
        $landlord_name = $user_detail->firstname.' '.$user_detail->lastname;

        $tenant_detail = User::findOrFail(Auth::user()->id);
        $tenant_name = $tenant_detail['firstname']." ".$tenant_detail['lastname'];
        
        $post['bid_id'] = $request->bid_id;
        $post['property_id'] = $request->property_id;
        $post['type_id'] = env('MESSAGE_TYPE_BID');
        
        $post['message'] = $tenant_name.' updated a bid against your property '.$request->property_name;
        $user_detail->notify(new \App\Notifications\PostNewNotification($post));

        /* Notifying landlord by email about bid submitted */
        $mail_result = $this->bid_submit($landlord_name, $landlord_email, $tenant_name, $property_name);

        Session::flash('Success','Bid was successfully updated');
        return redirect()->back();
    }

    public function survey(Request $request)
    {
        $survey = new Survey();
        $survey->user_id = Auth::user()->id;
        $survey->parent_id = Session::get('survey_parent');
        $survey->model_id = env('BIDDING');
        $survey->result = $request->surveyvalue;
        if(Session::has('survey_parent')){
            $survey->save();
            session()->forget('survey_parent');
            Session::flash('Success','Thank You for your feedback. Have a great day! :)');            
        }
        else{
            Session::flash('Error','Something went wrong !');            
        }        
        return redirect('biddings/bids');
    }


    /**
     * Deleting landlord note
     *
     * @param int $id
     *
     * @return Illuminate\View\View
     */
    public function deleteNote($note_id)
    {
        $delete = Note::where('id', $note_id)->delete();
        if($delete)
            Session::flash('Success','Note deleted Successfully');
        else
            Session::flash('Error','Something went wrong');
        return redirect()->back();
    }

    /**
     * Update landlord note
     *
     * @param int $id
     *
     * @return Illuminate\View\View
     */
    public function updateNote(Request $request)
    {
        $note_id = $request->note_id;
        $request->validate(['note'=>'required','visible'=>'required']);

        $note_details = Note::findOrFail($note_id)->toArray();
        if(!empty($note_details)){
            $visibile = $request->visible;
            $content = $request->note;

            $note_data['id'] = $note_id;
            $note_data['visibility_id'] = $visibile;
            $note_data['content'] = $content;

            $note->update($note_data);

            Session::flash('Success','Note updated Successfully');
        }
        else{
            Session::flash('Error','Something went wrong');
        }
            
        return redirect()->back();
    }


     # Admin section for bidding
    
    /**
     * Bid Listing in admin Section
     */
    public function adminBidList()
    {
        $tenant_id = 0;
        $bid_status_id = 0;
        $property_id = 0;
        $start_date = '';
        $end_date = '';
        $bid_id = 0;

        $conditions = array();

        if($_SERVER['REQUEST_METHOD']=='POST'){            
            $tenant_id = (isset($_POST['tenant_id'])) ? $_POST['tenant_id'] : 0;
            $bid_status_id = (isset($_POST['bid_status_id'])) ? $_POST['bid_status_id'] : 0;
            $property_id = (isset($_POST['property_id'])) ? $_POST['property_id'] : 0;
            $bid_id = (isset($_POST['bid_id'])) ? $_POST['bid_id'] : 0;
            $start_date = (!empty($_POST['start_date'])) ? date("Y-m-d", strtotime($_POST['start_date'])) : '';
            $end_date = (!empty($_POST['end_date'])) ? date("Y-m-d", strtotime($_POST['end_date'])) : '';
        }

        /* Getting filter parameters for pagination */
        if(isset($_GET['tenant_id']) && !empty($_GET['tenant_id']))
            $tenant_id = $_GET['tenant_id'];
        if(isset($_GET['bid_status_id']) && !empty($_GET['bid_status_id']))
            $bid_status_id = $_GET['bid_status_id'];
        if(isset($_GET['bid_id']) && !empty($_GET['bid_id']))
            $bid_id = $_GET['bid_id'];
        if(isset($_GET['property_id']) && !empty($_GET['property_id']))
            $property_id = $_GET['property_id'];
        if(isset($_GET['start_date']) && !empty($_GET['start_date']))
            $start_date = date("Y-m-d", strtotime($_GET['start_date']));
        if(isset($_GET['end_date']) && !empty($_GET['end_date']))
            $end_date = date("Y-m-d", strtotime($_GET['end_date']));
        /* Ends Here */

        //Cache::forget('tenant');
        /* Caching query */
        $tenants = Cache::remember('tenant', env('CACHE_TIME'), function()
        {
            return User::select('id','firstname','lastname','email')->where('user_type_id',env('USER_TYPE_TENANT'))->orderBy('firstname')->get()->toArray();
        });

        $bid_status = Cache::remember('bid_status', env('CACHE_TIME'), function()
        {
            return AppCode::bidstatus()->orderBy('name')->pluck('name','id')->all();
        });
        /* Cache ends here */

        $properties = Property::orderBy('name')->pluck('name','id')->all();
        
        $bids = Bidding::with(
                            array(
                                'Tenant'=>function($query){
                                    $query->select('id','firstname','lastname');
                                },
                                'EmploymentStatus'=>function($query){
                                    $query->select('id','name');
                                },
                                'BidStatus'=>function($query){
                                    $query->select('id','name');
                                }, 
                                'Property'=>function($query){
                                    $query->select('id','address','name');
                                }, 
                            )
                        )
                        //->orderBy('id','desc')
                        ->when($tenant_id, function ($query) use ($tenant_id) {
                            return $query->where('tenant_id',$tenant_id);
                        })
                        ->when($bid_status_id, function ($query) use ($bid_status_id) {
                            return $query->where('bid_status_id',$bid_status_id);
                        })
                         ->when($bid_id, function ($query) use ($bid_id) {
                            return $query->where('id',$bid_id);
                        })
                        ->when($property_id, function ($query) use ($property_id) {
                            return $query->where('property_id',$property_id);
                        })
                        ->when($start_date, function ($query) use ($start_date,$end_date) {
                            return $query->whereBetween('biddings.created_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                        })
                        ->sortable(['id'=>'desc'])
                        ->paginate(10);
        return view('admin/bids',compact('bids','tenants','bid_status','tenant_id','bid_status_id','properties','property_id','start_date','end_date'));
    }
    

    /**
     * Exporting Bids to Excel
     *
     * @param 
     *
     * @return Illuminate\View\View
     */
    public function adminExportBids($tenant_id,$bid_status_id,$property_id,$start_date=null,$end_date=null)
    {
        $conditions = array();

        if(!empty($start_date))
            $start_date = date("Y-m-d", strtotime($start_date));
        if(!empty($end_date))
            $end_date = date("Y-m-d", strtotime($end_date));

        $bids = Bidding::with(
                array(
                    'Tenant'=>function($query){
                        $query->select('id','firstname','lastname','email','mobile','address','city');
                    },
                    'EmploymentStatus'=>function($query){
                        $query->select('id','name');
                    },
                    'BidStatus'=>function($query){
                        $query->select('id','name');
                    }, 
                    'Property'=>function($query){
                        $query->select('id','address','name');
                    }, 
                )
            )
            ->orderBy('id','desc')
            ->when($tenant_id, function ($query) use ($tenant_id) {
                return $query->where('tenant_id',$tenant_id);
            })
            ->when($bid_status_id, function ($query) use ($bid_status_id) {
                return $query->where('bid_status_id',$bid_status_id);
            })
            ->when($property_id, function ($query) use ($property_id) {
                return $query->where('property_id',$property_id);
            })
            ->when($start_date, function ($query) use ($start_date,$end_date) {
                return $query->whereBetween('created_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
            })
            ->get()
            ->toArray();

        Excel::create('BidsReport', function($excel) use ($bids){
            $excel->sheet('bids', function($sheet) use ($bids){                
                $sheet->row(1,array('Bid Id','Property Name','Property Address','Tenant','Email','Mobile','Address','City','Proposed Rent','Move In Date','Tenant Note','Employment Status','Landlord Note','Bid Status','Bid Selection Time','Created On'));
               
                $i=2;
                foreach($bids as $bid){
                    $sheet->row($i, array($bid['id'], $bid['property']['name'], $bid['property']['address'], $bid['tenant']['firstname'].' '.$bid['tenant']['lastname'], $bid['tenant']['email'], $bid['tenant']['mobile'], $bid['tenant']['address'], $bid['tenant']['city'], $bid['propose_rent'], $bid['move_in_date'], $bid['tenant_note'], $bid['employment_status']['name'], $bid['landlord_note'], $bid['bid_status']['name'], $bid['bid_selection_time'],$bid['created_at']));
                    $i++;
                }
            });
        })->export('xls');
        die();
    }


    /**
     * Getting particular bid details
     *
     * @param int $id
     *
     * @return Illuminate\View\View
     */
    

public function adminBidForm($bid_id = null)
    {   
    if(isset($bid_id) && (int)$bid_id){
        $bidding_detail = Bidding::with('BidStatus','Tenant','Note')->findOrFail($bid_id)->toArray();
                return response()->json($bidding_detail);
         }  else { 
        echo 'error handling goes here';
        }    
        
        
    }



    /**
     * Users Bid History in admin Section
     */
    public function adminUsersBidHistory()
    {
        $start_date = $end_date = '';
        $tenant_id = 0;

        if($_SERVER['REQUEST_METHOD']=='POST'){            
            $tenant_id = (isset($_POST['tenant_id'])) ? $_POST['tenant_id'] : 0;
            $start_date = (!empty($_POST['start_date'])) ? date("Y-m-d", strtotime($_POST['start_date'])) : '';
            $end_date = (!empty($_POST['end_date'])) ? date("Y-m-d", strtotime($_POST['end_date'])) : '';
        }

        /* Getting filter parameters for pagination */
        if(isset($_GET['tenant_id']) && !empty($_GET['tenant_id']))
            $tenant_id = $_GET['tenant_id'];
        if(isset($_GET['start_date']) && !empty($_GET['start_date']))
            $start_date = date("Y-m-d", strtotime($_GET['start_date']));
        if(isset($_GET['end_date']) && !empty($_GET['end_date']))
            $end_date = date("Y-m-d", strtotime($_GET['end_date']));
        /* Ends Here */

        /* Caching query */
        $tenants = Cache::remember('tenant', env('CACHE_TIME'), function()
        {
            return User::select('id','firstname','lastname','email')->where('user_type_id',env('USER_TYPE_TENANT'))->orderBy('firstname')->get()->toArray();
        });


        $query1 = DB::table('biddings')         
                            ->selectRaw('0 as Total_Bids, count( bid_status_id) as Selected ,0 as Rejected ,0 as Completed ,0 as Active, 0 as Opted_Out, tenant_id')
                            ->where('bid_status_id',48)
                            ->when($tenant_id, function ($query) use ($tenant_id) {
                                return $query->where('tenant_id',$tenant_id);
                            })
                            ->when($start_date, function ($query) use ($start_date,$end_date) {
                                return $query->whereBetween('updated_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                            })
                            ->groupBy('tenant_id');

        $query2 = DB::table('biddings')         
                            ->selectRaw('0 as Total_Bids, 0 as Selected ,count( bid_status_id) as Rejected ,0 as Completed ,0 as Active, 0 as Opted_Out, tenant_id')
                            ->where('bid_status_id',49)
                            ->when($tenant_id, function ($query) use ($tenant_id) {
                                return $query->where('tenant_id',$tenant_id);
                            })
                            ->when($start_date, function ($query) use ($start_date,$end_date) {
                                return $query->whereBetween('updated_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                            })
                            ->groupBy('tenant_id');

        $query3 = DB::table('biddings')         
                            ->selectRaw('0 as Total_Bids, 0 as Selected ,0 as Rejected ,count( bid_status_id) as Completed ,0 as Active, 0 as Opted_Out, tenant_id')
                            ->where('bid_status_id',54)
                            ->when($tenant_id, function ($query) use ($tenant_id) {
                                return $query->where('tenant_id',$tenant_id);
                            })
                            ->when($start_date, function ($query) use ($start_date,$end_date) {
                                return $query->whereBetween('updated_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                            })
                            ->groupBy('tenant_id');

        $query4 = DB::table('biddings')         
                            ->selectRaw('0 as Total_Bids, 0 as Selected ,0 as Rejected ,0 as Completed ,count( bid_status_id) as Active, 0 as Opted_Out, tenant_id')
                            ->where('bid_status_id',56)
                            ->orWhere('bid_status_id',46)
                            ->when($tenant_id, function ($query) use ($tenant_id) {
                                return $query->where('tenant_id',$tenant_id);
                            })
                            ->when($start_date, function ($query) use ($start_date,$end_date) {
                                return $query->whereBetween('updated_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                            })
                            ->groupBy('tenant_id');

        $query5 = DB::table('biddings')         
                            ->selectRaw('0 as Total_Bids, 0 as Selected ,0 as Rejected ,0 as Completed ,0 as Active, count( bid_status_id) as Opted_Out, tenant_id')
                            ->where('bid_status_id',55)
                            ->when($tenant_id, function ($query) use ($tenant_id) {
                                return $query->where('tenant_id',$tenant_id);
                            })
                            ->when($start_date, function ($query) use ($start_date,$end_date) {
                                return $query->whereBetween('updated_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                            })
                            ->groupBy('tenant_id');

        $query6 = DB::table('biddings')         
                            ->selectRaw('count( distinct(id)) as Total_Bids, 0 as Selected ,0 as Rejected ,0 as Completed ,0 as Active, 0 as Opted_Out, tenant_id')
                            ->when($tenant_id, function ($query) use ($tenant_id) {
                                return $query->where('tenant_id',$tenant_id);
                            })
                            ->when($start_date, function ($query) use ($start_date,$end_date) {
                                return $query->whereBetween('updated_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                            })
                            ->groupBy('tenant_id');     

        $allUnions = $query1->union($query2)->union($query3)->union($query4)->union($query5)->union($query6);

        $results =  DB::table(DB::raw("({$allUnions->toSql()}) as bidding_report"))
                            ->leftJoin('users', 'users.id', '=', 'bidding_report.tenant_id')
                            ->selectRaw('SUM(Total_Bids) as Total_Bids, SUM(Selected) as Selected, SUM(Rejected) as Rejected, SUM(Completed) as Completed, SUM(Active) as Active, SUM(Opted_Out) as Opted_Out, tenant_id, users.firstname, users.lastname, users.id as user_id')
                            ->mergeBindings($allUnions)                            
                            ->groupBy('tenant_id')
                            //->get()
                            //->toArray();
                            ->paginate(10);

        return view('admin/users-bid-history',compact('results','start_date','end_date','user_id','tenants','tenant_id'));
    }

    /**
     * Exporting Bids History to Excel
     *
     * @param 
     *
     * @return Illuminate\View\View
     */
    public function adminExportBidsHistory($tenant_id,$start_date=null,$end_date=null)
    {
        if(!empty($start_date))
            $start_date = date("Y-m-d", strtotime($start_date));
        if(!empty($end_date))
            $end_date = date("Y-m-d", strtotime($end_date));

        $query1 = DB::table('biddings')         
                            ->selectRaw('0 as Total_Bids,count( bid_status_id) as Selected ,0 as Rejected ,0 as Completed ,0 as Active, 0 as Opted_Out, tenant_id')
                            ->where('bid_status_id',48)
                            ->when($tenant_id, function ($query) use ($tenant_id) {
                                return $query->where('tenant_id',$tenant_id);
                            })
                            ->when($start_date, function ($query) use ($start_date,$end_date) {
                                return $query->whereBetween('updated_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                            })
                            ->groupBy('tenant_id');

        $query2 = DB::table('biddings')         
                            ->selectRaw('0 as Total_Bids,0 as Selected ,count( bid_status_id) as Rejected ,0 as Completed ,0 as Active, 0 as Opted_Out, tenant_id')
                            ->where('bid_status_id',49)
                            ->when($tenant_id, function ($query) use ($tenant_id) {
                                return $query->where('tenant_id',$tenant_id);
                            })
                            ->when($start_date, function ($query) use ($start_date,$end_date) {
                                return $query->whereBetween('updated_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                            })
                            ->groupBy('tenant_id');

        $query3 = DB::table('biddings')         
                            ->selectRaw('0 as Total_Bids,0 as Selected ,0 as Rejected ,count( bid_status_id) as Completed ,0 as Active, 0 as Opted_Out, tenant_id')
                            ->where('bid_status_id',54)
                            ->when($tenant_id, function ($query) use ($tenant_id) {
                                return $query->where('tenant_id',$tenant_id);
                            })
                            ->when($start_date, function ($query) use ($start_date,$end_date) {
                                return $query->whereBetween('updated_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                            })
                            ->groupBy('tenant_id');

        $query4 = DB::table('biddings')         
                            ->selectRaw('0 as Total_Bids,0 as Selected ,0 as Rejected ,0 as Completed ,count( bid_status_id) as Active, 0 as Opted_Out, tenant_id')
                            ->where('bid_status_id',56)
                            ->orWhere('bid_status_id',46)
                            ->when($tenant_id, function ($query) use ($tenant_id) {
                                return $query->where('tenant_id',$tenant_id);
                            })
                            ->when($start_date, function ($query) use ($start_date,$end_date) {
                                return $query->whereBetween('updated_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                            })
                            ->groupBy('tenant_id');

        $query5 = DB::table('biddings')         
                            ->selectRaw('0 as Total_Bids, 0 as Selected ,0 as Rejected ,0 as Completed ,0 as Active, count( bid_status_id) as Opted_Out, tenant_id')
                            ->where('bid_status_id',55)
                            ->when($tenant_id, function ($query) use ($tenant_id) {
                                return $query->where('tenant_id',$tenant_id);
                            })
                            ->when($start_date, function ($query) use ($start_date,$end_date) {
                                return $query->whereBetween('updated_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                            })
                            ->groupBy('tenant_id');     

        $query6 = DB::table('biddings')         
                            ->selectRaw('count( id) as Total_Bids, 0 as Selected ,0 as Rejected ,0 as Completed ,0 as Active, 0 as Opted_Out, tenant_id')
                            ->when($tenant_id, function ($query) use ($tenant_id) {
                                return $query->where('tenant_id',$tenant_id);
                            })
                            ->when($start_date, function ($query) use ($start_date,$end_date) {
                                return $query->whereBetween('updated_at',[date("Y-m-d", strtotime($start_date)),date("Y-m-d", strtotime($end_date."+1 day"))]);
                            })
                            ->groupBy('tenant_id');

        $allUnions = $query1->union($query2)->union($query3)->union($query4)->union($query5)->union($query6);

        $results =  DB::table(DB::raw("({$allUnions->toSql()}) as bidding_report"))
                            ->leftJoin('users', 'users.id', '=', 'bidding_report.tenant_id')
                            ->selectRaw('SUM(Total_Bids) as Total_Bids, SUM(Selected) as Selected, SUM(Rejected) as Rejected, SUM(Completed) as Completed, SUM(Active) as Active, SUM(Opted_Out) as Opted_Out, tenant_id, users.firstname, users.lastname, users.id as user_id')
                            ->mergeBindings($allUnions)                            
                            ->groupBy('tenant_id')
                            ->get()
                            ->toArray();                            

        Excel::create('BidsHistoryReport', function($excel) use ($results){
            $excel->sheet('bids_history', function($sheet) use ($results){                
                $sheet->row(1,array('Tenant ID', 'Tenant Name','Total Bids', 'Selected Bids', 'Rejected Bids', 'Closed Bids', 'Active Bids', 'Opted-out Bids'));
               
                $i=2;
                foreach($results as $result){
                    $sheet->row($i, array($result->tenant_id, $result->firstname.' '.$result->lastname, $result->Total_Bids, $result->Selected, $result->Rejected, $result->Completed, $result->Active, $result->Opted_Out));
                    $i++;
                }
            });
        })->export('xls');
        die();
    }

}